// WebSocket-based chat system for Lingoano
// Check if already initialized to prevent conflicts
if (window.websocketChatInitialized) {
  console.log('WebSocket Chat already initialized, skipping...');
} else {
  window.websocketChatInitialized = true;

let wsInstance = null;
let wsConnected = false;
let wsTopic = "liguana-chat";
let wsUserId = null;



// Initialize WebSocket connection
async function initializeWebSocket() {
  try {
    
    // Get topic from session storage (set by matching system)
    const getSess = (primary, legacy) => sessionStorage.getItem(primary) || sessionStorage.getItem(legacy);
    const sessionTopic = getSess('Lingoano_topic', 'liguana_topic');
    const sessionLanguage = getSess('Lingoano_language', 'liguana_language');
    const sessionLevel = getSess('Lingoano_level', 'liguana_level');
    
    if (sessionTopic) {
      wsTopic = sessionTopic;
      console.log("📋 Session topic kullanılıyor:", wsTopic);
    } else if (sessionLanguage && sessionLevel) {
      // Generate topic if not in session storage
      const combined = `${sessionLanguage}_${sessionLevel}_ŞEBNEM`;
      const encoder = new TextEncoder();
      const data = encoder.encode(combined);
      const hashBuffer = await crypto.subtle.digest('SHA-256', data);
      const hashArray = Array.from(new Uint8Array(hashBuffer));
      const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
      wsTopic = `liguana_${hashHex.substring(0, 16)}`;
      console.log("🔧 Topic oluşturuldu:", wsTopic);
    }

    // Generate unique user ID
    wsUserId = 'user_' + Math.random().toString(36).substr(2, 9);
    
    // Connect to WebSocket server
    wsInstance = new WebSocket('wss://lingoano.onrender.com/liguanadinlemeserver/chat');
    
    wsInstance.onopen = function() {
      console.log("✅ WebSocket bağlantısı kuruldu");
      wsConnected = true;
      
      // Join chat room
      wsInstance.send(JSON.stringify({
        type: 'join',
        topic: wsTopic,
        userId: wsUserId
      }));
      
      // Enable message input
      const messageInput = document.getElementById("messageInput");
      const sendButton = document.getElementById("sendButton");
      
      const lm = window.languageManager;
      const t = (key, fallback) => (lm && typeof lm.getText === 'function' && lm.getText(key)) || fallback || key;
      if (messageInput) {
        messageInput.disabled = false;
        messageInput.placeholder = t('type-message', 'Mesajınızı yazın...');
      }
      
      if (sendButton) {
        sendButton.disabled = false;
      }
      
      appendMessage(t('websocket-connected', 'WebSocket ağına bağlandınız. Artık mesaj gönderebilirsiniz!'), "system");
    };
    
    wsInstance.onmessage = function(event) {
      try {
        const data = JSON.parse(event.data);
        console.log("📨 Mesaj alındı:", data);
        
        switch(data.type) {
          case 'message':
            if (data.userId !== wsUserId) {
              appendMessage(data.text, "received");
            }
            break;
          case 'system':
            appendMessage(data.text, "system");
            break;
          case 'user_joined':
            appendMessage(`${data.userId} sohbete katıldı`, "system");
            break;
          case 'user_left':
            appendMessage(`${data.userId} sohbetten ayrıldı`, "system");
            break;
        }
      } catch (err) {
        console.error("Mesaj çözümleme hatası:", err);
      }
    };
    
    wsInstance.onerror = function(error) {
      console.error("❌ WebSocket hatası:", error);
      enableDemoMode();
    };
    
    wsInstance.onclose = function() {
      console.log("WebSocket bağlantısı kapandı");
      wsConnected = false;
      enableDemoMode();
    };
    
  } catch (error) {
    console.error("❌ WebSocket başlatma hatası:", error);
    enableDemoMode();
  }
}

// Enable demo mode when WebSocket fails
function enableDemoMode() {
  const messageInput = document.getElementById("messageInput");
  const sendButton = document.getElementById("sendButton");
  
  const lm = window.languageManager;
  const t = (key, fallback) => (lm && typeof lm.getText === 'function' && lm.getText(key)) || fallback || key;
  if (messageInput) {
    messageInput.disabled = false;
    messageInput.placeholder = t('demo-mode-type-message', 'Demo modu - Mesajınızı yazın...');
  }
  
  if (sendButton) {
    sendButton.disabled = false;
  }
  
  // Enable demo mode
  window.demoMode = true;
  appendMessage(t('websocket-demo-mode', 'WebSocket bağlantısı kurulamadı. Demo modu aktif.'), "system");
}

// Send message via WebSocket
async function sendMessageInternal(text) {
  if (!text || !text.trim()) {
    console.warn("Mesaj gönderilemez: Metin boş");
    return false;
  }

  // Demo mode handling
  if (window.demoMode) {
    console.log("📤 Demo modu - Mesaj gönderildi:", text.trim());
    appendMessage(text.trim(), "sent");
    
    // Simulate received message after a short delay
    setTimeout(() => {
      const responses = [
        "Merhaba! Nasılsın?",
        "Bu dil öğrenme uygulaması gerçekten harika!",
        "Seninle konuşmak çok güzel.",
        "Hangi konularda pratik yapmak istiyorsun?",
        "Ben de bu dili öğrenmeye çalışıyorum.",
        "Birlikte pratik yapalım!",
        "Bu cümleyi nasıl söylersin?",
        "Çok güzel ilerliyorsun!",
        "Başka bir konuya geçelim mi?",
        "Bu kelimeyi nasıl telaffuz edersin?"
      ];
      const randomResponse = responses[Math.floor(Math.random() * responses.length)];
      appendMessage(randomResponse, "received");
    }, 1000 + Math.random() * 2000);
    
    return true;
  }

  // Real WebSocket mode
  if (!wsInstance || !wsConnected) {
    console.warn("Mesaj gönderilemez: WebSocket bağlı değil");
    return false;
  }

  try {
    const message = {
      type: 'message',
      topic: wsTopic,
      userId: wsUserId,
      text: text.trim(),
      timestamp: Date.now()
    };
    
    wsInstance.send(JSON.stringify(message));
    console.log("📤 Mesaj gönderildi:", message);
    appendMessage(text.trim(), "sent");
    return true;
    
  } catch (err) {
    console.error("❌ Mesaj gönderilemedi:", err);
    appendMessage("Mesaj gönderilemedi. Bağlantınızı kontrol edin.", "system");
    return false;
  }
}

// Add message to chat window
function appendMessage(text, type) {
  const chatMessages = document.getElementById("chatMessages");
  if (!chatMessages) {
    console.error("Chat messages container bulunamadı");
    return;
  }

  const messageEl = document.createElement("div");
  messageEl.className = `message ${type}`;
  messageEl.textContent = text;
  
  // Add timestamp for non-system messages
  if (type !== "system") {
    const timestamp = new Date().toLocaleTimeString('tr-TR', { 
      hour: '2-digit', 
      minute: '2-digit' 
    });
    messageEl.setAttribute('data-time', timestamp);
  }
  
  chatMessages.appendChild(messageEl);
  
  // Auto-scroll to bottom
  chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Make appendMessage globally available
window.appendMessage = appendMessage;

// Handle form submission
function setupMessageForm() {
  const messageForm = document.getElementById("messageForm");
  const messageInput = document.getElementById("messageInput");
  
  if (messageForm && messageInput) {
    // Handle form submission
    messageForm.addEventListener("submit", async function(e) {
      e.preventDefault();
      
      const text = messageInput.value.trim();
      if (text) {
        const success = await sendMessageInternal(text);
        if (success) {
          messageInput.value = "";
        }
      }
    });
    
    // Handle Enter key
    messageInput.addEventListener("keypress", async function(e) {
      if (e.key === "Enter" && !e.shiftKey) {
        e.preventDefault();
        
        const text = messageInput.value.trim();
        if (text) {
          const success = await sendMessageInternal(text);
          if (success) {
            messageInput.value = "";
          }
        }
      }
    });
    
    console.log("✅ Form event listeners kuruldu");
  } else {
    console.warn("⚠️ Form veya input elementi bulunamadı");
  }
}

// Cleanup on page unload
function cleanup() {
  if (wsInstance && wsConnected) {
    try {
      wsInstance.send(JSON.stringify({
        type: 'leave',
        topic: wsTopic,
        userId: wsUserId
      }));
      wsInstance.close();
    } catch (err) {
      console.error("Cleanup hatası:", err);
    }
  }
}

// Make cleanup globally available
window.cleanup = cleanup;

// Initialize WebSocket protocol
async function initializeWebSocketProtocol() {
  console.log("🦎 Lingoano Chat WebSocket başlatılıyor...");
  
  // Setup form handlers
  setupMessageForm();
  
  // Start WebSocket and wait for it to complete
  await initializeWebSocket();
  
  // Add welcome message
  setTimeout(() => {
    const role = sessionStorage.getItem('liguana_role');
    const language = sessionStorage.getItem('liguana_language');
    const level = sessionStorage.getItem('liguana_level');
    
    if (role && language && level) {
      appendMessage(`${language} (${level}) seviyesinde eşleşme bulundu! Sohbete hoş geldiniz.`, "system");
    } else {
      appendMessage("Sohbete hoş geldiniz! WebSocket üzerinden güvenli mesajlaşma.", "system");
    }
  }, 1000);
}

// Auto-initialize if not loaded by protocol selector
if (!window.protocolSelectorLoaded) {
  document.addEventListener('DOMContentLoaded', function() {
    initializeWebSocketProtocol();
    window.addEventListener('beforeunload', cleanup);
  });
}

// Make initialization function globally available
window.initializeWebSocketProtocol = initializeWebSocketProtocol;

// Make sendMessage globally available for backward compatibility
window.sendMessage = async function() {
  const messageInput = document.getElementById("messageInput");
  if (messageInput) {
    const text = messageInput.value.trim();
    if (text) {
      const success = await sendMessageInternal(text);
      if (success) {
        messageInput.value = "";
      }
    }
  }
};

} // Close the if statement from the beginning 