// Automatic Protocol Selector for Liguana Chat
// Automatically selects the best available protocol: IPFS -> WebRTC -> Demo Mode

let currentProtocol = 'ipfs'; // Default to IPFS

// Protocol configurations
const protocols = {
  webrtc: {
    name: 'WebRTC',
    description: 'Doğrudan peer-to-peer bağlantı',
    script: 'script/chat_webrtc.js',
    active: false
  },
  websocket: {
    name: 'WebSocket',
    description: 'Sunucu tabanlı gerçek zamanlı mesajlaşma',
    script: 'script/chat_websocket.js',
    active: true
  },
  ipfs: {
    name: 'IPFS',
    description: 'Merkezi olmayan peer-to-peer mesajlaşma',
    script: 'script/chat_ipfs.js',
    active: false
  }
};

// Initialize automatic protocol selector
function initializeProtocolSelector() {
  console.log('🔄 Otomatik Protokol Seçici başlatılıyor...');
  
  // Set flag to prevent auto-initialization
  window.protocolSelectorLoaded = true;
  
  // Check if IPFS library is available and enable it as primary
  if (typeof window.Ipfs !== 'undefined') {
    protocols.ipfs.active = true;
    console.log('✅ IPFS kütüphanesi mevcut (birincil olarak etkin)');
  } else {
    protocols.ipfs.active = false;
    console.log('⚠️ IPFS kütüphanesi bulunamadı');
  }
  
  // WebRTC is available in modern browsers
  protocols.webrtc.active = typeof window.RTCPeerConnection !== 'undefined';
  console.log('✅ WebRTC mevcut');
  
  // Start with IPFS (primary), fallback to WebRTC then WebSocket
  if (protocols.ipfs.active) {
    updateProtocolStatus('ipfs');
    startWithIPFS();
  } else if (protocols.webrtc.active) {
    updateProtocolStatus('webrtc');
    startWithWebRTC();
  } else {
    updateProtocolStatus('websocket');
    startWithWebSocket();
  }
}

// Update protocol status display
function updateProtocolStatus(protocol) {
  const statusElement = document.getElementById('protocolStatus');
  if (!statusElement) return;
  
  const statusIcon = statusElement.querySelector('.status-icon');
  const statusText = statusElement.querySelector('.status-text');
  
  const lm = window.languageManager;
  const t = (key, fallback) => (lm && typeof lm.getText === 'function' && lm.getText(key)) || fallback || key;
  if (protocol === 'ipfs') {
    statusIcon.textContent = '🔗';
    statusText.textContent = 'IPFS';
  } else if (protocol === 'webrtc') {
    statusIcon.textContent = '🌐';
    statusText.textContent = 'WebRTC';
  } else if (protocol === 'websocket') {
    statusIcon.textContent = '🛰️';
    statusText.textContent = 'WebSocket';
  } else if (protocol === 'error') {
    statusIcon.textContent = '❌';
    statusText.textContent = t('error', 'Hata');
  }
}

// Start with IPFS protocol
async function startWithIPFS() {
  console.log('🔗 IPFS protokolü ile başlatılıyor...');
  currentProtocol = 'ipfs';
  updateProtocolStatus('ipfs');
  try {
    await loadProtocol('ipfs');
  } catch (error) {
    console.error('❌ IPFS yükleme hatası:', error);
    // Fallback to WebRTC, then WebSocket
    if (protocols.webrtc.active) {
      startWithWebRTC();
    } else {
      startWithWebSocket();
    }
  }
}

// Start with WebRTC protocol
async function startWithWebRTC() {
  console.log('🌐 WebRTC protokolü ile başlatılıyor...');
  currentProtocol = 'webrtc';
  updateProtocolStatus('webrtc');
  
  try {
    await loadProtocol('webrtc');
  } catch (error) {
    console.error('❌ WebRTC yükleme hatası:', error);
    // Fallback to WebSocket
    startWithWebSocket();
  }
}

// Start with WebSocket protocol
async function startWithWebSocket() {
  console.log('🔌 WebSocket protokolü ile başlatılıyor...');
  currentProtocol = 'websocket';
  updateProtocolStatus('websocket');
  try {
    await loadProtocol('websocket');
  } catch (error) {
    console.error('❌ WebSocket yükleme hatası:', error);
    showConnectionError();
  }
}

// Show connection error
function showConnectionError() {
  console.log('❌ Bağlantı hatası');
  currentProtocol = 'error';
  updateProtocolStatus('error');
  
  const messageInput = document.getElementById("messageInput");
  const sendButton = document.getElementById("sendButton");
  
  if (messageInput) {
    messageInput.disabled = true;
    messageInput.placeholder = "Bağlantı hatası - Yeniden deneyin";
  }
  
  if (sendButton) {
    sendButton.disabled = true;
  }
  
  // Add error message
  setTimeout(() => {
    const chatMessages = document.getElementById('chatMessages');
    if (chatMessages) {
      const messageEl = document.createElement('div');
      messageEl.className = 'message system error';
      messageEl.textContent = 'Bağlantı hatası oluştu. Lütfen sayfayı yenileyin.';
      chatMessages.appendChild(messageEl);
      chatMessages.scrollTop = chatMessages.scrollHeight;
    }
  }, 500);
}

// Load protocol script
async function loadProtocol(protocol) {
  const protocolConfig = protocols[protocol];
  if (!protocolConfig || !protocolConfig.active) {
    throw new Error(`Protokol ${protocol} aktif değil`);
  }
  
  console.log(`📦 ${protocolConfig.name} protokolü yükleniyor...`);
  
  const script = document.createElement('script');
  script.src = protocolConfig.script;
  script.defer = true;

  return new Promise((resolve, reject) => {
    script.onload = async () => {
      console.log(`✅ ${protocolConfig.name} script dosyası yüklendi.`);
      try {
        if (protocol === 'ipfs' && typeof window.initializeIPFSProtocol === 'function') {
          await window.initializeIPFSProtocol();
        } else if (protocol === 'webrtc' && typeof window.initializeWebRTCProtocol === 'function') {
          await window.initializeWebRTCProtocol();
        } else if (protocol === 'websocket' && typeof window.initializeWebSocketProtocol === 'function') {
           await window.initializeWebSocketProtocol();
        } else {
            return reject(new Error(`${protocol} için başlatma fonksiyonu bulunamadı.`));
        }

        appendSystemMessage(`${protocolConfig.name} protokolü ile bağlandınız.`);
        resolve();
      } catch (initError) {
        console.error(`❌ ${protocolConfig.name} protokolü başlatılamadı:`, initError);
        reject(initError);
      }
    };
    
    script.onerror = () => {
      console.error(`❌ ${protocolConfig.name} protokolü scripti yüklenemedi`);
      reject(new Error(`${protocolConfig.name} protokolü scripti yüklenemedi`));
    };
    
    document.head.appendChild(script);
  });
}

function appendSystemMessage(message) {
    const chatMessages = document.getElementById('chatMessages');
    if (chatMessages) {
      const messageEl = document.createElement('div');
      messageEl.className = 'message system';
      messageEl.textContent = message;
      chatMessages.appendChild(messageEl);
      chatMessages.scrollTop = chatMessages.scrollHeight;
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
  console.log('🦎 Liguana Otomatik Protokol Seçici başlatılıyor...');
  
  // Wait a bit for other scripts to load
  setTimeout(() => {
    initializeProtocolSelector();
  }, 100);
});

// Make functions globally available
window.getCurrentProtocol = () => currentProtocol;
window.getProtocols = () => protocols; 