// Çok Dilli Günlük Kelime API - Geliştirilmiş Versiyon
class WordOfDayAPI {
    constructor() {
        // JSON dil dosyaları (eng.json formatı) için eşleme
        this.languageToFileMap = {
            en: 'eng.json',
            tr: 'tr.json',
            de: 'de.json',
            es: 'es.json',
            fr: 'fr.json',
            ru: 'ru.json'
        };

        // JSON yüklenemezse gösterilecek küçük varsayılan listeler
        this.defaultLists = {
            en: [
                { word: 'apple', meaning: ['a round fruit with red or green skin'], level: 'beginner' },
                { word: 'improve', meaning: ['make or become better'], level: 'intermediate' },
                { word: 'resilience', meaning: ['the capacity to recover quickly from difficulties'], level: 'advanced' }
            ],
            tr: [
                { word: 'elma', meaning: ['kırmızı veya yeşil kabuklu yuvarlak bir meyve'], level: 'beginner' },
                { word: 'geliştirmek', meaning: ['daha iyi hale getirmek'], level: 'intermediate' },
                { word: 'direnç', meaning: ['zorluklara karşı çabuk toparlanma yetisi'], level: 'advanced' }
            ],
            de: [
                { word: 'Apfel', meaning: ['eine runde Frucht mit roter oder grüner Schale'], level: 'beginner' },
                { word: 'verbessern', meaning: ['besser machen oder werden'], level: 'intermediate' },
                { word: 'Widerstandskraft', meaning: ['die Fähigkeit, sich schnell zu erholen'], level: 'advanced' }
            ],
            es: [
                { word: 'manzana', meaning: ['una fruta redonda con piel roja o verde'], level: 'beginner' },
                { word: 'mejorar', meaning: ['hacer o volverse mejor'], level: 'intermediate' },
                { word: 'resiliencia', meaning: ['capacidad de recuperarse rápidamente de las dificultades'], level: 'advanced' }
            ],
            fr: [
                { word: 'pomme', meaning: ['un fruit rond à peau rouge ou verte'], level: 'beginner' },
                { word: 'améliorer', meaning: ['rendre ou devenir meilleur'], level: 'intermediate' },
                { word: 'résilience', meaning: ["capacité à se remettre rapidement des difficultés"], level: 'advanced' }
            ],
            ru: [
                { word: 'яблоко', meaning: ['круглый плод с красной или зелёной кожурой'], level: 'beginner' },
                { word: 'улучшать', meaning: ['делать лучше или становиться лучше'], level: 'intermediate' },
                { word: 'устойчивость', meaning: ['способность быстро восстанавливаться после трудностей'], level: 'advanced' }
            ]
        };

        // Dizisel başlangıç tarihi (ilk kelime günü): 20 Ağustos 2025 (Yerel)
        this.sequenceStartLocal = new Date(2025, 7, 20); // 0-index month

        // Cache versiyonlu anahtar (güne duyarlı V3)
        this.cacheKey = 'wordOfDayCacheV3';
        this.cacheExpiry = 24 * 60 * 60 * 1000; // 24 saat (yedek mekanizma)
        try {
            const script = document.querySelector('script[src$="word-of-day.js"]');
            this.scriptBase = script ? (new URL('./', script.src)).href : null;
        } catch (e) {
            this.scriptBase = null;
        }
    }

    // Cache'den veri al
    getFromCache() {
        try {
            const cached = localStorage.getItem(this.cacheKey);
            if (cached) {
                const data = JSON.parse(cached);
                // Gün değiştiyse cache otomatik geçersiz
                const todayKey = this.getDayIndexFromStart();
                const sameDay = data && typeof data.dayKey === 'number' && data.dayKey === todayKey;
                const notExpired = Date.now() - data.timestamp < this.cacheExpiry;
                if (sameDay && notExpired && data.words) return data.words;
                // Farklı güne aitse temizle
                try { localStorage.removeItem(this.cacheKey); } catch (_) {}
            }
        } catch (error) {
            console.warn('Cache okuma hatası:', error);
        }
        return null;
    }

    // Cache'e veri kaydet
    saveToCache(words) {
        try {
            const cacheData = { words, timestamp: Date.now(), dayKey: this.getDayIndexFromStart() };
            localStorage.setItem(this.cacheKey, JSON.stringify(cacheData));
        } catch (error) {
            console.warn('Cache kaydetme hatası:', error);
        }
    }

    // Eski cache sürümlerini ve farklı güne ait veriyi temizle
    ensureFreshCache() {
        try {
            // V2 anahtarını da temizle (gün uyumsuzluklarını önlemek için)
            try { localStorage.removeItem('wordOfDayCacheV2'); } catch (_) {}
        } catch (_) {}
        // getFromCache zaten gün kontrolü yapıyor; burada sadece tetikliyoruz
        this.getFromCache();
    }

    // Dil listesi JSON'unu yükle (eng.json formatı)
    async loadLanguageList(language) {
        const file = this.languageToFileMap[language] || this.languageToFileMap.en;
        const buildCandidates = () => {
            const urls = [];
            if (this.scriptBase) {
                try { urls.push(new URL(file, this.scriptBase).href); } catch (e) {}
            }
            const path = (typeof location !== 'undefined') ? location.pathname : '';
            if (path.includes('/templates/')) {
                urls.push(`/templates/script/${file}`);
            }
            // Based on current page URL
            try { urls.push(new URL(`script/${file}`, location.href).href); } catch (e) {}
            try { urls.push(new URL(`./script/${file}`, location.href).href); } catch (e) {}
            urls.push(`script/${file}`);
            urls.push(`./script/${file}`);
            urls.push(`/script/${file}`);
            urls.push(`templates/script/${file}`);
            return Array.from(new Set(urls));
        };
        const tried = [];
        for (const url of buildCandidates()) {
            try {
                tried.push(url);
                const res = await fetch(url, { cache: 'no-store' });
                if (!res.ok) continue;
                const json = await res.json();
                const firstKey = Object.keys(json)[0];
                const list = Array.isArray(json[firstKey]) ? json[firstKey] : [];
                if (list.length > 0) {
                    console.info('WordOfDay JSON loaded:', { language, url });
                    return list;
                }
            } catch (e) {
                // try next
            }
        }
        console.warn('Dil listesi yüklenemedi, denenen yollar:', { language, tried });
        return [];
    }

    // 20.08.2025'ten bugüne kadar geçen gün sayısı (yerel güne göre)
    getDayIndexFromStart() {
        const now = new Date();
        const todayLocal = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        const startLocal = new Date(this.sequenceStartLocal.getFullYear(), this.sequenceStartLocal.getMonth(), this.sequenceStartLocal.getDate());
        const diffMs = todayLocal - startLocal;
        return Math.max(0, Math.floor(diffMs / (24 * 60 * 60 * 1000)));
    }

    // Günün kelimesini al (rastgele değil, tarihe göre sıra)
    async getWordOfDay(language) {
        // Gün geçişi çakışmalarını önlemek için cache tazele
        this.ensureFreshCache();
        const cached = this.getFromCache();
        if (cached && cached[language] && cached[language].word && cached[language].word !== 'N/A') return cached[language];

        // Dil listesi JSON'dan
        let list = await this.loadLanguageList(language);
        if (!list || list.length === 0) {
            list = await this.loadLanguageList('en');
        }
        if (!list || list.length === 0) {
            // Yerel varsayılan listelerden deneyelim
            list = this.defaultLists[language] || this.defaultLists.en || [];
        }
        if (!list || list.length === 0) {
            return { word: 'N/A', phonetic: '', meaning: 'No data', partOfSpeech: '', examples: [], origin: '', difficulty: 'Beginner' };
        }

        // Başlangıçtan itibaren index (25 Ağustos = 6. kelime)
        const dayIndex = this.getDayIndexFromStart();
        const idx = list.length > 0 ? (dayIndex % list.length) : 0;
        const entry = list[idx];

        const difficultyMap = { beginner: 'Beginner', intermediate: 'Intermediate', advanced: 'Advanced' };
        const normalized = {
            word: entry.word,
            phonetic: '',
            meaning: Array.isArray(entry.meaning) ? entry.meaning[0] : (entry.meaning || ''),
            partOfSpeech: '',
            examples: [],
            origin: '',
            difficulty: difficultyMap[(entry.level || '').toLowerCase()] || 'Beginner'
        };

        const allWords = cached || {};
        allWords[language] = normalized;
        this.saveToCache(allWords);
        return normalized;
    }

    // Tüm dillerin kelimelerini al (cache eksikse tamamla)
    async getAllWordsOfDay() {
        // Gün geçişi çakışmalarını önlemek için cache tazele
        this.ensureFreshCache();
        const languages = ['en', 'tr', 'de', 'es', 'fr', 'ru'];
        const cached = this.getFromCache();
        const result = cached ? { ...cached } : {};

        await Promise.all(
            languages.map(async lang => {
                const needsRefresh = !result[lang] || !result[lang].word || result[lang].word === 'N/A';
                if (needsRefresh) {
                    result[lang] = await this.getWordOfDay(lang);
                }
            })
        );

        this.saveToCache(result);
        return result;
    }

    // Dil adını döndür
    getLanguageName(code) {
        const names = {
            en: 'İngilizce',
            tr: 'Türkçe',
            de: 'Almanca',
            es: 'İspanyolca',
            fr: 'Fransızca',
            ru: 'Rusça'
        };
        return names[code] || code;
    }

    // Zorluk rengi
    getDifficultyColor(difficulty) {
        const colors = {
            Beginner: '#4CAF50',
            Intermediate: '#FF9800',
            Advanced: '#F44336'
        };
        return colors[difficulty] || '#666';
    }

    // API kontrolü kaldırıldı (yerel JSON kullanılacak)
}

// Global instance
window.wordOfDayAPI = new WordOfDayAPI();
