const hakkindaTranslations = {
    tr: {
      feature1_title: "Çevrimdışı Gramer Dersleri",
      feature1_desc: "İnternet olmadan dilin temel kurallarını öğrenin. İnteraktif dersler ve anlaşılır örneklerle gramer bilginizi sağlam temeller üzerine kurun. Karmaşık yapıları bile kolayca anlayın ve cümle kurma becerinizi geliştirerek kendinize olan güveninizi her an, her yerde artırın.",
      feature2_title: "Geniş Kelime Havuzu",
      feature2_desc: "Seviyenize uygun binlerce kelimeye çevrimdışı erişin. Akıllı bilgi kartları ve pratik alıştırmalarla yeni kelimeleri kalıcı olarak öğrenin. Kelime dağarcığınızı hızla zenginleştirin ve kendinizi daha akıcı ve çeşitli ifadelerle anlatmanın keyfini çıkarın.",
      feature3_title: "Telaffuz Pratiği",
      feature3_desc: "Anadili konuşanların telaffuzlarını dinleyin ve kendi sesinizi kaydederek karşılaştırın. Gelişmiş ses tanıma teknolojimizle hatalarınızı anında tespit edin. Aksanınızı düzeltin, daha net ve doğal bir konuşma tarzı benimseyerek iletişim becerilerinizi güçlendirin.",
      feature4_title: "Dinamik Sohbet Odaları",
      feature4_desc: "Dünyanın dört bir yanından dil öğrenenlerle canlı sohbetlere katılın. Teorik bilgilerinizi gerçek zamanlı konuşma pratiğine dökün. Yeni kültürler keşfedin, uluslararası dostluklar kurun ve konuşma korkunuzu yenerek akıcılığınızı bir üst seviyeye taşıyın.",
      feature5_title: "Eğlenceli Görevler ve Bulmacalar",
      feature5_desc: "Öğrenmeyi, her gün yenilenen kelime bulmacaları ve dil oyunları ile eğlenceli bir maceraya dönüştürün. Puanlar toplayıp seviyeler atlayarak motivasyonunuzu yüksek tutun. Dil öğrenimini bir alışkanlık haline getirin ve sıkılmadan ilerlemenin keyfini çıkarın."
    },
    en: {
      feature1_title: "Offline Grammar Lessons",
      feature1_desc: "Learn the basic rules of the language without internet. Build your grammar knowledge on a solid foundation with interactive lessons and clear examples. Understand even complex structures easily and boost your confidence by improving your sentence-building skills anytime, anywhere.",
      feature2_title: "Extensive Vocabulary Pool",
      feature2_desc: "Get offline access to thousands of words suited to your level. Learn new words permanently with smart flashcards and practical exercises. Rapidly enrich your vocabulary and enjoy expressing yourself with more fluency and a greater variety of phrases.",
      feature3_title: "Pronunciation Practice",
      feature3_desc: "Listen to native speakers' pronunciations and compare by recording your own voice. Instantly detect your mistakes with our advanced voice recognition technology. Correct your accent and adopt a clearer, more natural way of speaking to strengthen your communication skills.",
      feature4_title: "Dynamic Chat Rooms",
      feature4_desc: "Join live conversations with language learners from all over the world. Put your theoretical knowledge into real-time speaking practice. Discover new cultures, build international friendships, and overcome your fear of speaking to take your fluency to the next level.",
      feature5_title: "Fun Tasks and Puzzles",
      feature5_desc: "Turn learning into a fun adventure with daily updated word puzzles and language games. Keep your motivation high by collecting points and leveling up. Make language learning a habit and enjoy progressing without getting bored."
    },
    de: {
      feature1_title: "Offline-Grammatiklektionen",
      feature1_desc: "Lernen Sie die Grundregeln der Sprache ohne Internet. Bauen Sie Ihr Grammatikwissen mit interaktiven Lektionen und klaren Beispielen auf einer soliden Basis auf. Verstehen Sie selbst komplexe Strukturen leicht und stärken Sie Ihr Selbstvertrauen durch bessere Satzbildung.",
      feature2_title: "Großer Wortschatzpool",
      feature2_desc: "Erhalten Sie Offline-Zugriff auf Tausende von Wörtern, die Ihrem Niveau entsprechen. Lernen Sie neue Wörter dauerhaft mit intelligenten Karteikarten und praktischen Übungen. Erweitern Sie schnell Ihren Wortschatz und genießen Sie es, sich flüssiger und vielfältiger auszudrücken.",
      feature3_title: "Ausspracheübung",
      feature3_desc: "Hören Sie sich die Aussprache von Muttersprachlern an und vergleichen Sie sie, indem Sie Ihre eigene Stimme aufnehmen. Erkennen Sie Ihre Fehler sofort mit unserer fortschrittlichen Spracherkennungstechnologie. Korrigieren Sie Ihren Akzent für einen klareren und natürlicheren Sprechstil.",
      feature4_title: "Dynamische Chaträume",
      feature4_desc: "Nehmen Sie an Live-Gesprächen mit Sprachlernern aus der ganzen Welt teil. Setzen Sie Ihr theoretisches Wissen in die Praxis um. Entdecken Sie neue Kulturen, schließen Sie internationale Freundschaften und überwinden Sie Ihre Sprechangst, um Ihre Sprachkenntnisse zu verbessern.",
      feature5_title: "Unterhaltsame Aufgaben und Rätsel",
      feature5_desc: "Verwandeln Sie das Lernen in ein Abenteuer mit täglich neuen Worträtseln und Sprachspielen. Halten Sie Ihre Motivation hoch, indem Sie Punkte sammeln und aufsteigen. Machen Sie das Sprachenlernen zur Gewohnheit und genießen Sie den Fortschritt, ohne sich zu langweilen."
    },
    es: {
      feature1_title: "Lecciones de Gramática sin Conexión",
      feature1_desc: "Aprende las reglas básicas del idioma sin internet. Construye tu conocimiento de la gramática sobre una base sólida con lecciones interactivas y ejemplos claros. Entiende fácilmente incluso las estructuras complejas y aumenta tu confianza mejorando tu habilidad para construir frases.",
      feature2_title: "Amplio Banco de Vocabulario",
      feature2_desc: "Accede sin conexión a miles de palabras adecuadas a tu nivel. Aprende nuevas palabras de forma permanente con tarjetas inteligentes y ejercicios prácticos. Enriquece rápidamente tu vocabulario y disfruta expresándote con más fluidez y una mayor variedad de frases.",
      feature3_title: "Práctica de Pronunciación",
      feature3_desc: "Escucha la pronunciación de hablantes nativos y compárala grabando tu propia voz. Detecta tus errores al instante con nuestra avanzada tecnología de reconocimiento de voz. Corrige tu acento y adopta un estilo de habla más claro y natural para fortalecer tus habilidades de comunicación.",
      feature4_title: "Salas de Chat Dinámicas",
      feature4_desc: "Únete a conversaciones en vivo con estudiantes de idiomas de todo el mundo. Pon en práctica tus conocimientos teóricos en conversaciones en tiempo real. Descubre nuevas culturas, haz amistades internacionales y supera tu miedo a hablar para llevar tu fluidez al siguiente nivel.",
      feature5_title: "Tareas y Puzles Divertidos",
      feature5_desc: "Convierte el aprendizaje en una aventura divertida con puzles de palabras y juegos de idiomas actualizados a diario. Mantén tu motivación alta acumulando puntos y subiendo de nivel. Haz del aprendizaje de idiomas un hábito y disfruta progresando sin aburrirte."
    },
    fr: {
      feature1_title: "Leçons de Grammaire Hors Ligne",
      feature1_desc: "Apprenez les règles de base de la langue sans Internet. Bâtissez vos connaissances en grammaire sur des bases solides avec des leçons interactives et des exemples clairs. Comprenez facilement même les structures complexes et renforcez votre confiance en améliorant votre syntaxe.",
      feature2_title: "Vaste Banque de Vocabulaire",
      feature2_desc: "Accédez hors ligne à des milliers de mots adaptés à votre niveau. Apprenez de nouveaux mots de manière permanente avec des fiches mémoire intelligentes et des exercices pratiques. Enrichissez rapidement votre vocabulaire et exprimez-vous avec plus de fluidité et de variété.",
      feature3_title: "Pratique de la Prononciation",
      feature3_desc: "Écoutez la prononciation des locuteurs natifs et comparez-la en enregistrant votre propre voix. Détectez instantanément vos erreurs grâce à notre technologie avancée de reconnaissance vocale. Corrigez votre accent et adoptez un style oral plus clair et naturel.",
      feature4_title: "Salons de Discussion Dynamiques",
      feature4_desc: "Rejoignez des conversations en direct avec des apprenants du monde entier. Mettez vos connaissances théoriques en pratique lors de conversations en temps réel. Découvrez de nouvelles cultures, nouez des amitiés internationales et surmontez votre peur de parler pour progresser.",
      feature5_title: "Tâches et Puzzles Amusants",
      feature5_desc: "Transformez l'apprentissage en une aventure amusante avec des puzzles de mots et des jeux de langue mis à jour quotidiennement. Gardez votre motivation élevée en accumulant des points et en montant de niveau. Faites de l'apprentissage une habitude et progressez sans vous ennuyer."
    },
    ru: {
      feature1_title: "Офлайн-уроки грамматики",
      feature1_desc: "Изучайте основные правила языка без интернета. Создайте прочную основу грамматических знаний с помощью интерактивных уроков и понятных примеров. Легко разбирайтесь даже в сложных конструкциях и повышайте свою уверенность, улучшая навыки построения предложений.",
      feature2_title: "Обширный словарный запас",
      feature2_desc: "Получите офлайн-доступ к тысячам слов, соответствующих вашему уровню. Учите новые слова навсегда с помощью умных карточек и практических упражнений. Быстро пополняйте свой словарный запас и наслаждайтесь возможностью выражать свои мысли более свободно и разнообразно.",
      feature3_title: "Практика произношения",
      feature3_desc: "Слушайте произношение носителей языка и сравнивайте, записывая свой голос. Мгновенно выявляйте ошибки с помощью нашей передовой технологии распознавания речи. Исправляйте свой акцент, говорите чётче и естественнее, укрепляя свои коммуникативные навыки.",
      feature4_title: "Динамические чаты",
      feature4_desc: "Присоединяйтесь к живым беседам с изучающими языки со всего мира. Применяйте теоретические знания на практике в реальном времени. Открывайте для себя новые культуры, заводите друзей из других стран и преодолевайте страх говорить, чтобы достичь нового уровня беглости.",
      feature5_title: "Увлекательные задания и головоломки",
      feature5_desc: "Превратите обучение в увлекательное приключение с ежедневно обновляемыми словесными головоломками и языковыми играми. Поддерживайте высокую мотивацию, зарабатывая очки и повышая уровень. Сделайте изучение языка привычкой и наслаждайтесь прогрессом без скуки."
    }
  };

// Language management class specifically for hakkında.html
class HakkindaLanguageManager {
    constructor(translations) {
        this.translations = translations;
        this.currentLanguage = localStorage.getItem('Lingoano-site-language') || 'tr'; // Default to 'tr'
        this.init();
    }
    
    init() {
        // Set the language on page load
        this.setLanguage(this.currentLanguage);
        this.bindEvents();
    }
    
    setLanguage(lang, saveToStorage = false) {
        if (!this.translations[lang]) {
            console.warn(`Language ${lang} not found in hakkında translations, falling back to Turkish`);
            lang = 'tr';
        }
        
        this.currentLanguage = lang;
        
        if (saveToStorage) {
            localStorage.setItem('Lingoano-site-language', lang);
        }
        
        // Update language selectors on the page
        const languageSelects = document.querySelectorAll('.site-language-select');
        languageSelects.forEach(select => {
            if (select) {
                select.value = lang;
            }
        });
        
        this.updatePageContent();
    }
    
    getText(key) {
        const langTranslations = this.translations[this.currentLanguage] || this.translations['tr'];
        return langTranslations[key] || key;
    }
    
    updatePageContent() {
        const elements = document.querySelectorAll('[data-lang]');
        elements.forEach(element => {
            const key = element.getAttribute('data-lang');
            // Only translate keys that exist in the 'hakkinda' translations
            if (this.translations[this.currentLanguage][key] || this.translations['tr'][key]) {
                const translation = this.getText(key);
                if (translation && translation !== key) {
                     if (element.tagName === 'INPUT' && element.hasAttribute('placeholder')) {
                        element.placeholder = translation;
                    } else if (element.hasAttribute('title')) {
                        element.title = translation;
                    } else if (element.hasAttribute('aria-label')) {
                        element.setAttribute('aria-label', translation);
                    } else if (element.hasAttribute('alt')) {
                        element.alt = translation;
                    } else {
                        element.innerHTML = translation;
                    }
                }
            }
        });

        // Update document title if it has data-lang
        const titleElement = document.querySelector('title[data-lang]');
        if (titleElement) {
            const key = titleElement.getAttribute('data-lang');
            if (this.translations[this.currentLanguage][key] || this.translations['tr'][key]) {
                const translation = this.getText(key);
                if (translation && translation !== key) {
                    titleElement.textContent = translation;
                }
            }
        }
    }

    bindEvents() {
        const languageSelects = document.querySelectorAll('.site-language-select');
        languageSelects.forEach(select => {
            if (select) {
                select.addEventListener('change', (e) => {
                    this.setLanguage(e.target.value, true);
                });
            }
        });
    }
}

// Initialize language manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Pass the specific translations to the manager
    new HakkindaLanguageManager(hakkindaTranslations);
});